#include "Mario.h"
#include "Level.h"

Mario::Mario()
{
	idleSmall.addFrame(sf::IntRect(80, 34, 16, 16));
	walkSmall.addFrame(sf::IntRect(97, 34, 16, 16));
	walkSmall.addFrame(sf::IntRect(114, 34, 16, 16));
	walkSmall.addFrame(sf::IntRect(131, 34, 16, 16));
	walkSmall.setFrameSpeed(1.f / 12.f);
	jumpSmall.addFrame(sf::IntRect(165, 34, 16, 16));
	duckSmall.addFrame(sf::IntRect(80, 34, 16, 16));
	slideSmall.addFrame(sf::IntRect(216, 34, 16, 16));

	idleBig.addFrame(sf::IntRect(80, 1, 16, 32));
	walkBig.addFrame(sf::IntRect(97, 1, 16, 32));
	walkBig.addFrame(sf::IntRect(114, 1, 16, 32));
	walkBig.addFrame(sf::IntRect(131, 1, 16, 32));
	walkBig.setFrameSpeed(1.f / 12.f);
	jumpBig.addFrame(sf::IntRect(165, 1, 16, 32));
	duckBig.addFrame(sf::IntRect(182, 1, 16, 32));
	slideBig.addFrame(sf::IntRect(216, 1, 16, 32));

	dead.addFrame(sf::IntRect(182, 34, 16, 16));		//Add frames from the Mario sprite sheet to all of our animations

	currentAnimation = &idleSmall;	//We initially will be idle and small Mario
	setTextureRect(currentAnimation->getCurrentFrame());

	setAirbourne(true);	//We spawn just above the first block, so we're airbourne
	setAlive(true);	//We've spawned in, so we're alive

	lives = 3;	//We start with 3 lives
	powerUpState = PowerUp::Big;	//We set our power up state to big here
	powerDown();					//so that when we call powerDown, it runs through its small Mario functions
	Level::audioManager->stopAllSounds();	//Call this to stop the power down sound playing
	setInvincible(false, 0.0);	//We stop the invincibilty frames from calling powerDown
}


Mario::~Mario()
{
}

void Mario::update(float dt)
{
	currentAnimation->animate(dt);
	setTextureRect(currentAnimation->getCurrentFrame());

	velocity.y += gravity * dt;

	if (!getAirbourne() && isAlive())	//If we're alive and not airbourne
	{
		setAnimForPowerUp(Anims::Idle);	//Set the idle animation for Mario's power up state
	}

	if (getInvincible())	//If we're invincible
	{
		if (Timer::setWaitTime(getInvincibilityTime(), dt))	//After the time is up, stop the invincibility frames
		{
			setInvincible(false, 0.0);
		}
	}

	if (getPosition().y > 768 + getSize().y && isAlive())	//If we've fallen into a pit
	{
		setPosition(getPosition().x, 1500);	//Set Mario's y outside the player's view so he doesn't fire back up visibly when playing his death animation (when he falls, his death animation doesn't play like when he hits an enemy - bit of a cheat here)
		kill();	//Kill Mario
	}

	move(velocity.x * dt, velocity.y * dt);	//Move Mario
	
}

void Mario::handleInput(float dt)
{
	if (isAlive() && inputsAllowed)	//If we're alive and allowed to input
	{
		if (!getAirbourne() && !input->isKeyDown(sf::Keyboard::Up))	//If we're not airbourne and we're not pressing Up
		{
			canJump = true;	//We're allowed to jump
		}

		if (input->isKeyDown(sf::Keyboard::Up))
		{
			if (!getAirbourne() && canJump)	//If we're not airbourne and allowed to jump
			{
				canJump = false;	//We can't jump again yet
				setAirbourne(true);	//We're jumping so we're airbourne
				velocity.y = (-47.f / 10.f) * scale;

				Level::audioManager->playSoundByName("Jump");
				setAnimForPowerUp(Anims::Jump);	//Set the correct jumping animation for Mario's power up
				currentAnimation->reset();
				setTextureRect(currentAnimation->getCurrentFrame());
				setCollisionForPowerUp(sf::FloatRect(0, 4, 50, 46), sf::FloatRect(0, 7, 50, 93));		//Set the jump animation collision boxes for big and small Mario
			}
		}

		if (input->isKeyDown(sf::Keyboard::Left))
		{
			velocity.x = -225;	//Move Mario to the left

			if (!getAirbourne())	//If we're not airbourne
			{
				setAnimForPowerUp(Anims::Walk);	//Set the correct walk animation for Mario's power up
				setCollisionForPowerUp(sf::FloatRect(0, 0, 50, 50), sf::FloatRect(0, 0, 50, 100));		//Set the walk animation collision boxes for big and small Mario
			}

			currentAnimation->setFlipped(true);	//Flip Mario to face the other direction
		}

		if (input->isKeyDown(sf::Keyboard::Right))
		{
			velocity.x = 225;	//Move Mario to the right

			if (!getAirbourne())	//If we're not airbourne
			{
				setAnimForPowerUp(Anims::Walk);	//Set the correct walk animation for Mario's power up
				setCollisionForPowerUp(sf::FloatRect(0, 0, 50, 50), sf::FloatRect(0, 0, 50, 100));		//Set the walk animation collision boxes for big and small Mario
			}

			currentAnimation->setFlipped(false);	//Un-flip Mario to face the correct direction
		}

		if (input->isKeyDown(sf::Keyboard::Down))
		{
			if (!getAirbourne())	//If we're not airbourne
			{
				velocity.x = 0;	//Kill all x velocity
				setAnimForPowerUp(Anims::Duck);	//Set the correct duck animation for Mario's power up
				currentAnimation->reset();
				setTextureRect(currentAnimation->getCurrentFrame());
			}
		}

		if (!input->isKeyDown(sf::Keyboard::Left) && !input->isKeyDown(sf::Keyboard::Right))	//If we're not pressing left or right
		{
			setVelocity(0, getVelocity().y);	//Kill Mario's x velocity
		}
	}
}

void Mario::setCollisionForPowerUp(sf::FloatRect small, sf::FloatRect big)	//setCollisionForPowerUp sets the correct size collision box for Mario's power up state
{
	if (powerUpState == PowerUp::Small)	//If Mario is small
	{
		this->setCollisionBox(small);	//Set the small collision box
	}

	else   //Else Mario is big
	{
		this->setCollisionBox(big);	//Set the big collision box
	}
}

void Mario::setAnimForPowerUp(Anims anim)	//setAnimForPowerUp sets the correct size animation for Mario's power up state
{
	bool shouldFlip = currentAnimation->getFlipped();	//Should we flip the new animation?

	if (powerUpState == PowerUp::Big)	//If we're big we set the big animations
	{
		switch (anim)
		{
			case Idle:
				currentAnimation = &idleBig;
				break;

			case Walk:
				currentAnimation = &walkBig;
				break;

			case Jump:
				currentAnimation = &jumpBig;
				break;

			case Duck:
				currentAnimation = &duckBig;
				break;

			case Slide:
				currentAnimation = &slideBig;
				break;
		}
	}

	else   //Else if we're small we set the small animations
	{
		switch (anim)
		{
			case Idle:
				currentAnimation = &idleSmall;
				break;

			case Walk:
				currentAnimation = &walkSmall;
				break;

			case Jump:
				currentAnimation = &jumpSmall;
				break;

			case Duck:
				currentAnimation = &duckSmall;
				break;

			case Slide:
				currentAnimation = &slideSmall;
				break;
		}
	}

	currentAnimation->setFlipped(shouldFlip);	//Set the animation flip to shouldFlip
}

bool Mario::handleEnemyCollision(MySprite* enemy)
{
	if (getPosition().y <= enemy->getPosition().y && getAirbourne() && enemy->getSpriteType() == SpriteType::GoombaSprite)	//If we're above the enemy, we're airbourne and the enemy is a goomba
	{
		return true;	//The goomba will die, not Mario so return true
	}

	else if (enemy->getSpriteType() == SpriteType::KoopaSprite)	//If the enemy is a Koopa
	{
		return true;	//Immediately return true as the Koopa's collision handler will control it and Mario
	}

	else    //Else Mario will die
	{
		return false;	//So we return false
	}
}

void Mario::addScore(int scoreToAdd)
{
	score += scoreToAdd;	//Increment private score by scoreToAdd
}

void Mario::removeScore(int scoreToRemove)
{
	score -= scoreToRemove;	//Decrement private score by scoreToRemove
}

int Mario::getScore()
{
	return score;	//Return private score
}

void Mario::addCoin()
{
	Level::audioManager->playSoundByName("CollectCoin");
	addScore(200);	//Every time we collect a coin, we gain 200 score

	coins++;	//Increment coin count

	if (coins >= 10)	//If we have 10 or more coins
	{
		addLife();	//We gain a one up
		removeCoins();	//We remove the coins to start collecting from the beginning
	}
}

void Mario::removeCoins()
{
	coins = 0;
}

int Mario::getCoins()
{
	return coins;	//Return private coins
}

void Mario::addLife()
{
	Level::audioManager->playSoundByName("1UP");
	lives++;	//Increment Mario's lives
}

int Mario::getLives()
{
	return lives;	//Return private lives
}

void Mario::setLives(int life)
{
	lives = life;	//Set private lives to life
}

void Mario::setWon(bool won)
{
	haveWon = won;	//Set private haveWon to won
}

bool Mario::getWon()
{
	return haveWon;	//Return private haveWon
}

bool Mario::isFlipped()
{
	return currentAnimation->getFlipped();	//Return whether or not Mario's animation is currently flipped
}

bool Mario::isPoweredUp()
{
	if (powerUpState == PowerUp::Big)	//If Mario is powered up we return true
	{
		return true;
	}

	else   //Else Mario isn't powered up so we return false
	{
		return false;
	}
}

void Mario::kill()
{
	Level::audioManager->stopAllMusic();
	Level::audioManager->playSoundByName("Death");
	setAlive(false);	//We're dead
	inputsAllowed = false;	//The player loses control over Mario
	lives--;	//We lose a life
	setVelocity(sf::Vector2f(0, (-47.f / 10.f) * scale));	//We set the velocity here to begin Mario's death animation where he is fired into the air then falls back down
	currentAnimation = &dead;	//We set the current animation to dead
	setTextureRect(currentAnimation->getCurrentFrame());
}

void Mario::bounce()	//Bounce is called to make Mario bounce off an enemy after a collision
{
	setAnimForPowerUp(Anims::Jump);		//Set the jumping animation for Mario's power up state
	velocity.y = (-3.2f / 1.f) * scale;	//Set Mario to jump in the air
	setAirbourne(true);	//We're jumping, so we're airbourne
}

void Mario::powerUp()
{
	Level::audioManager->playSoundByName("PowerUp");

	if (!isPoweredUp())
	{
		setPosition(getPosition().x, getPosition().y - 50);	//Set Mario 50 pixels higher to accommodate for the size increase
	}

	powerUpState = PowerUp::Big;	//Mario is powered up
	setSize(sf::Vector2f(50, 100));	//We increase his size to accommodate the power up
	setCollisionForPowerUp(sf::FloatRect(0, 0, 50, 50), sf::FloatRect(0, 0, 50, 100));	//Set the correct collison box for Mario's power up state

	addScore(1000);	//We've powered up, so we add 1000 score
}

void Mario::powerDown()
{
	setInvincible(true, 2);	//We set Mario to be invincible for 2 seconds so he can't be powered down immediately again (gives the player time to escape the enemy)

	switch (powerUpState)
	{
		case PowerUp::Big:    //If we're currently big
			Level::audioManager->playSoundByName("PowerDown");
			powerUpState = PowerUp::Small;	//We turn small
			setSize(sf::Vector2f(50, 50));	//We decrease our size
			setPosition(getPosition().x, getPosition().y + 50);	//We set ourselves 50 y pixels lower to account for the size decrease
			break;

		case PowerUp::Small:   //If we're currently small
			kill();	//If we're getting powered down when we're small, we die, so call kill
			break;
	}

	setCollisionForPowerUp(sf::FloatRect(0, 0, 50, 50), sf::FloatRect(0, 0, 50, 100));	//Set the correct collision box for Mario's power up state
}

void Mario::slideDownPole()
{
	inputsAllowed = false;	//We're sliding down the pole so ignore all user input
	setAnimForPowerUp(Anims::Slide);	//Set the sliding animation for Mario's power up state
	setVelocity(sf::Vector2f(0, (3.f / 2.f) * scale));	//Kill Mario's x velocity, set his y velocity to send him downwards and slide down the pole

	if (getPosition().y >= 580)	//If we're at the bottom of the pole
	{
		setVelocity(sf::Vector2f(0, 0));	//Kill all velocity
	}
}

bool Mario::walkToCastle()
{
	setAirbourne(false);	//If we're walking, we're on the floor and not airbourne, so set airbourne to false
	velocity.x = 300;	//Move Mario to the right toward the castle
	setAnimForPowerUp(Anims::Walk);	//Set the walk animation for Mario's power up state

	if (getPosition().x >= 10935)	//If we've made it to the castle
	{
		setVelocity(0, 0);	//Kill all velocity
		setFillColor(sf::Color::Transparent);	//Set Mario to be transparent - looks like he walked into the castle but he's actually just invisible (bit of a cheat)
		return true;	//We're at the castle now so return true
	}

	return false;	//We've not made it to the castle yet, return false
}